# 画面設計書 8-Hash Codes

## 概要

本ドキュメントは、VBCorLibプロジェクトのサンプルアプリケーション「Hash Codes」の画面設計書です。

### 本画面の処理概要

Hash Codesは、VBCorLibライブラリが提供する各種ハッシュアルゴリズムをデモンストレーションする画面です。入力されたテキストに対して、SHA-1、SHA-256、SHA-384、SHA-512、MD5、RIPEMD-160の6種類のハッシュ値をリアルタイムで計算・表示します。

**業務上の目的・背景**：ハッシュ関数は、データの整合性検証、パスワードの安全な保存、デジタル署名などで広く使用される暗号技術の基盤です。VBCorLibは.NET Frameworkと同様のハッシュアルゴリズムAPIを提供し、VB6/VBAでも標準的なハッシュ処理が可能です。本画面は、各ハッシュアルゴリズムの出力長の違いや、入力変更時の即座のハッシュ値変化を視覚的に示すことを目的としています。

**画面へのアクセス方法**：Examples/Cryptography/HashCodes/HashCodes.vbpプロジェクトを開き、F5キーで実行することでこの画面が表示されます。

**主要な操作・処理内容**：
1. Sourceテキストボックスにハッシュ対象のテキストを入力
2. テキスト変更ごとに自動的に6種類のハッシュ値を計算
3. 各ハッシュ値を16進数文字列で表示

**画面遷移**：本画面は独立したデモアプリケーションであり、他の画面への遷移はありません。

**権限による表示制御**：本画面は開発者向けデモツールであり、特別な権限制御は実装されていません。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 55 | SHA1Managed | 主機能 | SHA-1ハッシュ計算（ComputeHash） |
| 56 | SHA256Managed | 主機能 | SHA-256ハッシュ計算（ComputeHash） |
| 57 | SHA384Managed | 主機能 | SHA-384ハッシュ計算（ComputeHash） |
| 58 | SHA512Managed | 主機能 | SHA-512ハッシュ計算（ComputeHash） |
| 59 | MD5CryptoServiceProvider | 主機能 | MD5ハッシュ計算（ComputeHash） |
| 60 | RIPEMD160Managed | 主機能 | RIPEMD-160ハッシュ計算（ComputeHash） |
| 42 | Encoding | 補助機能 | 入力文字列のバイト配列変換（UTF8.GetBytes） |
| 17 | StringBuilder | 補助機能 | ハッシュ値の16進数文字列表示（AppendFormat、AppendLine） |

## 画面種別

デモンストレーション / ハッシュ計算

## URL/ルーティング

VB6デスクトップアプリケーションのため、URLルーティングは存在しません。フォームファイル: `Examples/Cryptography/HashCodes/Form1.frm`

## 入出力項目

| 項目名 | コントロール | 項目種別 | データ型 | 説明 |
|--------|-------------|----------|----------|------|
| txtSource | TextBox | 入力 | String | ハッシュ対象のテキスト |

## 表示項目

| 項目名 | コントロール | データ型 | 説明 | ハッシュ長 |
|--------|-------------|----------|------|----------|
| txtSHA1 | TextBox | String | SHA-1ハッシュ値（16進数） | 160bit (20bytes) |
| txtSHA256 | TextBox | String | SHA-256ハッシュ値（16進数） | 256bit (32bytes) |
| txtSHA384 | TextBox | String | SHA-384ハッシュ値（16進数） | 384bit (48bytes) |
| txtSHA512 | TextBox | String | SHA-512ハッシュ値（16進数） | 512bit (64bytes) |
| txtMD5 | TextBox | String | MD5ハッシュ値（16進数） | 128bit (16bytes) |
| txtRIPEMD160 | TextBox | String | RIPEMD-160ハッシュ値（16進数） | 160bit (20bytes) |

## イベント仕様

### 1-txtSource_Change

ソーステキスト変更時にハッシュ計算を実行します。

**処理内容**:
1. ComputeHashesを呼び出して全ハッシュ値を計算

**コード参照**: 行194-196

```vb
Private Sub txtSource_Change()
    Call ComputeHashes(txtSource.Text)
End Sub
```

### 2-ComputeHashes（内部関数）

全ハッシュアルゴリズムでハッシュ値を計算します。

**処理内容**:
1. Encoding.UTF8.GetBytesで入力テキストをバイト配列に変換
2. 各ハッシュアルゴリズムのComputeHashでハッシュ計算
3. DisplayHashで結果を16進数表示

**コード参照**: 行199-209

```vb
Private Sub ComputeHashes(ByVal Text As String)
    Dim Bytes() As Byte
    Bytes = Encoding.UTF8.GetBytes(Text)

    Call DisplayHash(mSHA1.ComputeHash(Bytes), txtSHA1)
    Call DisplayHash(mSHA256.ComputeHash(Bytes), txtSHA256)
    Call DisplayHash(mSHA384.ComputeHash(Bytes), txtSHA384)
    Call DisplayHash(mSHA512.ComputeHash(Bytes), txtSHA512)
    Call DisplayHash(mMD5.ComputeHash(Bytes), txtMD5)
    Call DisplayHash(mRIPEMD160.ComputeHash(Bytes), txtRIPEMD160)
End Sub
```

### 3-DisplayHash（内部関数）

ハッシュ値を16進数文字列に変換して表示します。

**処理内容**:
1. StringBuilderで16進数文字列を構築
2. 16バイトごとに改行
3. 各バイトを{0:X2}形式で2桁16進数に変換

**コード参照**: 行211-222

```vb
Private Sub DisplayHash(ByRef Hash() As Byte, ByVal Box As TextBox)
    Dim sb As New StringBuilder
    sb.Length = 0

    Dim i As Long
    For i = 0 To UBound(Hash)
        If (i > 0) And ((i Mod 16) = 0) Then Call sb.AppendLine
        Call sb.AppendFormat("{0:X2}", Hash(i))
    Next i

    Box.Text = sb.ToString
End Sub
```

## データベース更新仕様

本画面はデータベースを使用しません。

## メッセージ仕様

本画面にメッセージ表示機能はありません。

## 例外処理

| 例外種別 | 対処方法 |
|---------|---------|
| エンコードエラー | VBランタイムエラー発生（通常発生しない） |

## 備考

- BorderStyle = Fixed Single（サイズ変更不可）
- MaxButton = False, MinButton = False（最大化・最小化ボタン無効）
- 全ハッシュオブジェクトはNew演算子でインスタンス化
- フォント: Courier 9.75pt（等幅フォントで16進数を見やすく）
- SHA-384、SHA-512はMultiLine=Trueで複数行表示

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ハッシュアルゴリズムオブジェクトの宣言を理解します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Form1.frm | `Examples/Cryptography/HashCodes/Form1.frm` | 行185-190: モジュールレベル変数宣言 |

**読解のコツ**: 全てのハッシュオブジェクトはNew演算子で宣言時に初期化されます。これにより、Form_Loadなしで即座に使用可能です。

```vb
Private mSHA1 As New SHA1CryptoServiceProvider
Private mSHA256 As New SHA256Managed
Private mSHA384 As New SHA384Managed
Private mSHA512 As New SHA512Managed
Private mMD5 As New MD5CryptoServiceProvider
Private mRIPEMD160 As New RIPEMD160Managed
```

#### Step 2: エントリーポイントを理解する

テキスト変更イベントがエントリーポイントです。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Form1.frm | `Examples/Cryptography/HashCodes/Form1.frm` | 行194-196: txtSource_Change |

**主要処理フロー**:
1. **行195**: ComputeHashes(txtSource.Text)を呼び出し

#### Step 3: ハッシュ計算処理を理解する

ハッシュ計算と表示のフローを確認します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Form1.frm | `Examples/Cryptography/HashCodes/Form1.frm` | 行199-209: ComputeHashes |
| 3-2 | Form1.frm | `Examples/Cryptography/HashCodes/Form1.frm` | 行211-222: DisplayHash |

**主要処理フロー（ComputeHashes）**:
- **行201**: Encoding.UTF8.GetBytesで文字列→バイト配列
- **行203-208**: 各ハッシュアルゴリズムでComputeHash

**主要処理フロー（DisplayHash）**:
- **行213**: StringBuilderを作成
- **行217**: 16バイトごとにAppendLineで改行
- **行218**: {0:X2}形式で2桁16進数変換
- **行221**: ToString()で文字列化してテキストボックスに表示

### プログラム呼び出し階層図

```
txtSource_Change
    │
    └─ ComputeHashes(Text)
           │
           ├─ Encoding.UTF8.GetBytes(Text)
           │      └─ Bytes()
           │
           ├─ mSHA1.ComputeHash(Bytes)
           │      └─ DisplayHash(Hash, txtSHA1)
           │
           ├─ mSHA256.ComputeHash(Bytes)
           │      └─ DisplayHash(Hash, txtSHA256)
           │
           ├─ mSHA384.ComputeHash(Bytes)
           │      └─ DisplayHash(Hash, txtSHA384)
           │
           ├─ mSHA512.ComputeHash(Bytes)
           │      └─ DisplayHash(Hash, txtSHA512)
           │
           ├─ mMD5.ComputeHash(Bytes)
           │      └─ DisplayHash(Hash, txtMD5)
           │
           └─ mRIPEMD160.ComputeHash(Bytes)
                  └─ DisplayHash(Hash, txtRIPEMD160)

DisplayHash(Hash, Box)
    │
    └─ StringBuilder
           ├─ AppendFormat("{0:X2}", Hash(i))
           ├─ AppendLine (16バイトごと)
           └─ ToString → Box.Text
```

### データフロー図

```
[入力]               [処理]                    [出力]

txtSource.Text ───▶ Encoding.UTF8.GetBytes ───▶ Bytes()
                          │
                          ├─▶ mSHA1.ComputeHash ────▶ txtSHA1
                          │
                          ├─▶ mSHA256.ComputeHash ──▶ txtSHA256
                          │
                          ├─▶ mSHA384.ComputeHash ──▶ txtSHA384
                          │
                          ├─▶ mSHA512.ComputeHash ──▶ txtSHA512
                          │
                          ├─▶ mMD5.ComputeHash ─────▶ txtMD5
                          │
                          └─▶ mRIPEMD160.ComputeHash▶ txtRIPEMD160
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Form1.frm | `Examples/Cryptography/HashCodes/Form1.frm` | ソース | メインフォーム |
| HashCodes.vbp | `Examples/Cryptography/HashCodes/HashCodes.vbp` | プロジェクト | プロジェクト定義 |
| VBCorLib.dll | 外部 | DLL | VBCorLibライブラリ本体 |
